------------------------------------------ Normierung des Aufbaus der Issue Messages ------------------------------------------

-- Normierung der Zusammengefassung aller Issues
CREATE OR REPLACE FUNCTION tvalidation.issue_list__build(
      _issue_list       jsonb[]
  ) RETURNS jsonb AS $$
      
      SELECT
        jsonb_build_object(
            'all_issues',
            _issue_list
        )
      ;
      
  $$ LANGUAGE sql IMMUTABLE;
--

-- Normierung des Issue Context
  -- Kontext enthält ausgeführte Subroutine (Funktion) und Optionen (Parameter) des einen Aspekts der Validierung.
CREATE OR REPLACE FUNCTION tvalidation.issue_context__build(
      _subroutine       varchar,
      _options          jsonb
  ) RETURNS jsonb AS $$
      
      SELECT
        jsonb_build_object(
            'context',
            jsonb_build_object(
                'subroutine', _subroutine,
                'options', _options
            )
        )
      ;
      
  $$ LANGUAGE sql IMMUTABLE;
--

-- Normierung des spez. Issues
  -- Mit Fehlschlagen der Validierung, wird der fehlgeschlagene Aspekt verzeichnet.
  -- Das Issue ist menschenlesbar.
  -- Der Hint ist menschenlesbar und sollte die Erwartungshaltung enthalten.
  -- Zugehörige Issue Objects werden zugeordnet.
CREATE OR REPLACE FUNCTION tvalidation.issue__build(
      _issue            varchar,
      _hint             varchar,
      VARIADIC _objects jsonb[]
  ) RETURNS jsonb AS $$
      
      SELECT
        jsonb_build_object(
            'issue', _issue,
            'hint', _hint,
            'objects', coalesce( _objects, '{null}'::jsonb[] )
        )
      ;
      
  $$ LANGUAGE sql IMMUTABLE;
--

-- Normierung des Issue Objects
  -- Objekte, die nicht den erwarteten Werten entsprechen.
  -- 1 Objekt ist (aktuell) ein Verweis auf 1 Datenzeile.
  -- Werden per tvalidation.issue__build einem Issue zugeordnet.
CREATE OR REPLACE FUNCTION tvalidation.issue_object__build(
      _table  varchar,
      _id     anyelement
  ) RETURNS jsonb AS $$
      
      SELECT
        jsonb_build_object(
            'table', _table,
            'id', _id
        )
      ;
      
  $$ LANGUAGE sql IMMUTABLE;
--


------------------------------------------------- Normierung der Validierungen -------------------------------------------------

-- Validierung des IC eines Artikels
CREATE OR REPLACE FUNCTION tvalidation.intcod__check(
      _ak_nr                    varchar,    -- Einstieg über Artikelnummer
      VARIADIC _expected_values integer[]   -- not nullable
  ) RETURNS boolean AS $$
  DECLARE
      _ic         integer;
      _is_valid   boolean;
  BEGIN
      
      -- Fehlerbehandlung: fehlende Parameter
      IF
              _ak_nr IS NULL
          OR  array_position( _expected_values, null ) IS NOT NULL
      THEN
          RAISE EXCEPTION '%', lang_text( 16885 );
      END IF;
      
      
      _ic := GetArtIc( _ak_nr );
      
      -- Prüfung
      _is_valid := _ic = any ( _expected_values ) IS TRUE;
      
      
      RETURN _is_valid;
  END $$ LANGUAGE plpgsql STABLE;
--

-- Validierung der Status
  -- normierte Enum-Liste
  -- meistens Positionen: stv.st_stat, op6.o6_stat, auftg.ag_stat
CREATE OR REPLACE FUNCTION tvalidation.stat__check(
      _stat                     varchar,    -- Einstieg über Status
      VARIADIC _expected_values varchar[]   -- nullable
  ) RETURNS boolean AS $$
  DECLARE
      _expected_value_is_nullable boolean;
      _stat_list                  varchar[];
      _is_valid                   boolean;
  BEGIN
      
      -- Es dürfen (nur) nulls enthalten sein.
      _expected_value_is_nullable := array_position( _expected_values, null ) IS NOT NULL;
      
      _stat_list := TSystem.ENUM_list_to_array( _stat );
      
      -- Prüfung
      IF
          -- Wert
          (
                  _stat IS NOT NULL
              AND _expected_values @> _stat_list
          )
          -- oder nullable
          OR (
                  _stat IS NULL
              AND _expected_value_is_nullable
          )
          
      THEN
          _is_valid := true;
      ELSE
          _is_valid := false;
      END IF;
      
      
      RETURN _is_valid;
  END $$ LANGUAGE plpgsql IMMUTABLE;
--


--------------------------------------------------- Stücklisten-Validierung ---------------------------------------------------

-- Validierung der Stücklisten-Typen
  -- jeder Typ kann mehrere Funktionen zugeordnet bekommen, die jeweils einen Aspekt prüfen (SRP).
CREATE TABLE tvalidation.stv_typ__subroutine (
  sttvld_id           serial PRIMARY KEY,
  sttvld_typ          varchar(10) NOT NULL REFERENCES stv_typ ON UPDATE CASCADE ON DELETE CASCADE,
  sttvld_order        integer NOT NULL,
  sttvld_subroutine   varchar NOT NULL, -- Funktion für einen Aspekt der Validierung (SRP)
  sttvld_options      jsonb NOT NULL,   -- zugehörige Funktionsparameter
  
  UNIQUE ( sttvld_typ, sttvld_subroutine, sttvld_options )
);

-- zentrale Validierungsfunktion
  -- auf Basis von Einträgen in tvalidation.stv_typ__subroutine
  -- und entspr. Funktionen (SRP)
CREATE OR REPLACE FUNCTION tvalidation.stv_typ__execute(
      _stv_id         integer,  -- Einstieg über stvzutxt.stv_id
      _debug          boolean = false,
      
      OUT _type       varchar,
      OUT _is_valid   boolean,
      OUT _issues     jsonb      
  ) RETURNS record AS $$
  DECLARE
      _validation           record;
      _validation_call      varchar;
      _validation_result    record;
      _issue_context        jsonb;
      _collected_issues     jsonb[];
  BEGIN
      
      -- Fehlerbehandlung: fehlende Parameter
      IF _stv_id IS NULL THEN
          RAISE EXCEPTION '%', lang_text( 16885 );
      END IF;
      
      _type :=
            stv_stat
          FROM stvzutxt
          WHERE stv_id = _stv_id
      ;
      
      FOR _validation IN
          
          SELECT
            sttvld_subroutine,
            sttvld_options
          FROM tvalidation.stv_typ__subroutine
          WHERE sttvld_typ = _type
          ORDER BY sttvld_order, sttvld_id
          
      LOOP
          -- ID ergänzen
          _validation.sttvld_options :=
              jsonb_set(
                  _validation.sttvld_options,
                  '{ _stv_id }',
                  to_jsonb( _stv_id )
              )
          ;
          
          _validation_call := null;
          
          -- Call der Validierungsfunktion mit allen Parametern (Übergabe per jsonb)
          _validation_call :=
              format(
                  'SELECT * FROM tvalidation.%I( %I => %L )',
                  _validation.sttvld_subroutine,
                  '_options',
                  _validation.sttvld_options
              )
          ;
          
          -- Debug-Ausgabe des Calls
          IF _debug THEN
              RAISE NOTICE 'validation_call: %', _validation_call;
          END IF;
          
          
          -- Validierung durchführen
          EXECUTE _validation_call INTO _validation_result;
          
          
          -- Beim Fehlschlagen einer Validierung, gilt die gesamte Validierung als fehlgeschlagen.
          IF _validation_result._is_valid IS FALSE THEN
              _is_valid := false;
          END IF;
          
          
          -- Issues mit ergänztem Context sammeln.
          IF _validation_result._issue IS NOT NULL THEN
              
              _issue_context := null;
              
              _issue_context :=
                  tvalidation.issue_context__build(
                      _validation.sttvld_subroutine,
                      _validation.sttvld_options
                  )
              ;
              
              _collected_issues := _collected_issues || ( _validation_result._issue || _issue_context );
              
          END IF;
          
      END LOOP;
      
      -- ohne Prüfung immer valide
      _is_valid := coalesce( _is_valid, true );
      
      -- Ausgabe aller Issues
      IF _collected_issues IS NOT NULL THEN
          _issues := tvalidation.issue_list__build( _collected_issues );
      END IF;
      
      
      RETURN;
  END $$ LANGUAGE plpgsql STABLE;
--

-- Validierung des IC der Stücklisten-Wurzel
CREATE OR REPLACE FUNCTION tvalidation.stv_typ__wurzel__intcod(
      _stv_id                   integer,    -- Einstieg über stvzutxt.stv_id
      VARIADIC _expected_values integer[],  -- not nullable
      
      OUT _is_valid   boolean,
      OUT _issue      jsonb
  ) RETURNS record AS $$
  DECLARE
      _ak_nr            varchar;
      _issue_object_art jsonb;
  BEGIN
      
      -- Fehlerbehandlung: fehlende Parameter
      IF
              _stv_id IS NULL
          OR  array_position( _expected_values, null ) IS NOT NULL
      THEN
          RAISE EXCEPTION '%', lang_text( 16885 );
      END IF;
      
      
      _ak_nr :=
            stv_zn
          FROM stvzutxt
          WHERE stv_id = _stv_id
      ;
      
      
      -- Validierung des IC
      IF _ak_nr IS NOT NULL THEN
          
          _is_valid := tvalidation.intcod__check( _ak_nr, VARIADIC _expected_values );
          
          -- Issue Objekt ergänzen
          IF _is_valid IS FALSE THEN
              _issue_object_art := tvalidation.issue_object__build( 'art', _ak_nr );
          END IF;
          
      END IF;
      
      
      -- ohne Prüfung nicht valide
      _is_valid := coalesce( _is_valid, false );
      
      
      -- Definition des Issues
      IF _is_valid IS FALSE THEN
          
          _issue :=
              tvalidation.issue__build(
                  -- IC des Artikels der Wurzel nicht korrekt.
                  lang_text( 16886 ),
                  -- IC ... wird/werden erwartet.
                  format(
                      lang_text( 16887 ),
                      array_to_string( _expected_values, ', ' )
                  ),
                  _issue_object_art
              )
          ;
          
      END IF;
      
      
      RETURN;
  END $$ LANGUAGE plpgsql STABLE;
--

-- Polymorph für Parameter per jsonb
  -- siehe tvalidation.stv_typ__execute
CREATE OR REPLACE FUNCTION tvalidation.stv_typ__wurzel__intcod(
      _options jsonb,
      
      OUT _is_valid   boolean,
      OUT _issue      jsonb
  ) RETURNS record AS $$
      
      SELECT
        tvalidation.stv_typ__wurzel__intcod(
            ( _options->>'_stv_id' )::int,
            VARIADIC ARRAY( SELECT jsonb_array_elements_text( _options->'_expected_values' )::int )
        )
      ;
      
  $$ LANGUAGE sql STABLE;
--


-- Validierung des IC der direkten inneren Knoten
CREATE OR REPLACE FUNCTION tvalidation.stv_typ__innere_knoten__intcod(
      _stv_id                   integer,    -- Einstieg über stvzutxt.stv_id
      VARIADIC _expected_values integer[],  -- not nullable
      
      OUT _is_valid   boolean,
      OUT _issue      jsonb
  ) RETURNS record AS $$
  DECLARE
      _stv_pos          record;
      _is_valid_pos     boolean;
      _issue_object_art jsonb;
      _issue_object_stv jsonb;
      _issue_objects    jsonb[];
  BEGIN
      
      -- Fehlerbehandlung: fehlende Parameter
      IF
              _stv_id IS NULL
          OR  array_position( _expected_values, null ) IS NOT NULL
      THEN
          RAISE EXCEPTION '%', lang_text( 16885 );
      END IF;
      
      
      -- relevante Daten der inneren Knoten ermitteln
      FOR _stv_pos IN
          
          SELECT
            stv.st_n AS ak_nr,
            stv.st_id
          FROM stvzutxt
            JOIN stv ON st_zn = stv_zn
          WHERE stv_id = _stv_id
          ORDER BY st_pos, st_id
          
      LOOP
          
          -- Initialisierung
          _is_valid := coalesce( _is_valid, true );
          _issue_object_art := null;
          _issue_object_stv := null;
          
          -- Validierung des IC
          _is_valid_pos := tvalidation.intcod__check( _stv_pos.ak_nr, VARIADIC _expected_values );
          
          
          IF _is_valid_pos IS FALSE THEN
              
              -- Beim Fehlschlagen einer Validierung, gilt die gesamte Validierung als fehlgeschlagen.
              _is_valid := false;
              
              
              -- Issue Objekte ergänzen
              _issue_object_art := tvalidation.issue_object__build( 'art', _stv_pos.ak_nr );
              
              _issue_object_stv := tvalidation.issue_object__build( 'stv', _stv_pos.st_id );
              
              _issue_objects := _issue_objects || _issue_object_art || _issue_object_stv;
              
          END IF;
          
      END LOOP;
      
      
      -- ohne Prüfung nicht valide
      _is_valid := coalesce( _is_valid, false );
      
      
      -- Definition des Issues
      IF _is_valid IS FALSE THEN
          
          _issue :=
              tvalidation.issue__build(
                  -- IC des Artikels des/der inneren Knoten nicht korrekt.
                  lang_text( 16888 ),
                  -- IC ... wird/werden erwartet.
                  format(
                      lang_text( 16887 ),
                      array_to_string( _expected_values, ', ' )
                  ),
                  VARIADIC _issue_objects
              )
          ;
          
      END IF;
      
      
      RETURN;
  END $$ LANGUAGE plpgsql STABLE;
--

-- Polymorph für Parameter per jsonb
  -- siehe tvalidation.stv_typ__execute
CREATE OR REPLACE FUNCTION tvalidation.stv_typ__innere_knoten__intcod(
      _options jsonb,
      
      OUT _is_valid   boolean,
      OUT _issue      jsonb
  ) RETURNS record AS $$
      
      SELECT
        tvalidation.stv_typ__innere_knoten__intcod(
            ( _options->>'_stv_id' )::int,
            VARIADIC ARRAY( SELECT jsonb_array_elements_text( _options->'_expected_values' )::int )
        )
      ;
      
  $$ LANGUAGE sql STABLE;
--


-- Validierung des Stat P. der direkten inneren Knoten
CREATE OR REPLACE FUNCTION tvalidation.stv_typ__innere_knoten__st_stat(
      _stv_id                   integer,    -- Einstieg über stvzutxt.stv_id
      VARIADIC _expected_values varchar[],  -- nullable
      
      OUT _is_valid   boolean,
      OUT _issue      jsonb
  ) RETURNS record AS $$
  DECLARE
      _stv_pos                    record;
      _is_valid_pos               boolean;
      _issue_object_stv           jsonb;
      _issue_objects              jsonb[];
  BEGIN
      
      -- Fehlerbehandlung: fehlende Parameter
      IF _stv_id IS NULL THEN
          RAISE EXCEPTION '%', lang_text( 16885 );
      END IF;
      
      
      -- relevante Daten der inneren Knoten ermitteln
      FOR _stv_pos IN
          
          SELECT
            stv.st_stat,
            stv.st_id
          FROM stvzutxt
            JOIN stv ON st_zn = stv_zn
          WHERE stv_id = _stv_id
          ORDER BY st_pos, st_id
          
      LOOP
          
          -- Initialisierung
          _is_valid := coalesce( _is_valid, true );
          _issue_object_stv := null;
          
          -- Validierung der Status
          _is_valid_pos := tvalidation.stat__check( _stv_pos.st_stat, VARIADIC _expected_values );
          
          
          IF _is_valid_pos IS FALSE THEN
              
              -- Beim Fehlschlagen einer Validierung, gilt die gesamte Validierung als fehlgeschlagen.
              _is_valid := false;
              
              
              -- Issue Objekte ergänzen
              _issue_object_stv := tvalidation.issue_object__build( 'stv', _stv_pos.st_id );
              
              _issue_objects := _issue_objects || _issue_object_stv;
              
          END IF;
          
      END LOOP;
      
      
      -- ohne Prüfung nicht valide
      _is_valid := coalesce( _is_valid, false );
      
      
      -- Definition des Issues
      IF _is_valid IS FALSE THEN
          
          _issue :=
              tvalidation.issue__build(
                  -- Stat P. der Stücklistenposition(en) nicht korrekt.
                  lang_text( 16889 ),
                  -- Stat P. ... wird/werden erwartet.
                  format(
                      lang_text( 16890 ),
                      array_to_string( _expected_values, ', ', '<null>' )
                  ),
                  VARIADIC _issue_objects
              )
          ;
          
      END IF;
      
      
      RETURN;
  END $$ LANGUAGE plpgsql STABLE;
--

-- Polymorph für Parameter per jsonb
  -- siehe tvalidation.stv_typ__execute
CREATE OR REPLACE FUNCTION tvalidation.stv_typ__innere_knoten__st_stat(
      _options jsonb,
      
      OUT _is_valid   boolean,
      OUT _issue      jsonb
  ) RETURNS record AS $$
      
      SELECT
        tvalidation.stv_typ__innere_knoten__st_stat(
            ( _options->>'_stv_id' )::int,
            VARIADIC ARRAY( SELECT jsonb_array_elements_text( _options->'_expected_values' )::varchar )
        )
      ;
      
  $$ LANGUAGE sql STABLE;
--


--------------------------------------------------- weitere Validierung ---------------------------------------------------
